# -*- coding: utf-8 -*-

"""
/***************************************************************************
 ContourGenerator
                                 A QGIS plugin
 Generates contours from point layer
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2018-04-24
        copyright            : (C) 2018 by Chris Crook
        email                : ccrook@linz.govt.nz
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

__author__ = 'Chris Crook'
__date__ = '2018-04-24'
__copyright__ = '(C) 2018 by Chris Crook'

# This will get replaced with a git SHA1 when you do a git archive

__revision__ = '$Format:%H$'

import os.path
from PyQt5.QtCore import QCoreApplication, QUrl
from PyQt5.QtGui import QIcon
from qgis.core import (QgsProcessing,
                       QgsFeatureSink,
                       QgsProcessingAlgorithm,
                       QgsProcessingParameterFeatureSource,
                       QgsProcessingParameterEnum,
                       QgsProcessingParameterExpression,
                       QgsProcessingParameterNumber,
                       QgsProcessingParameterBoolean,
                       QgsProcessingParameterString,
                       QgsProcessingParameterFeatureSink,
                       QgsWkbTypes)
from .ContourGenerator import ContourGenerator, ContourType, ContourExtendOption
from .ContourGenerator import ContourError, ContourMethodError
from . import ContourMethod
from . import resources


def tr(string):
    return QCoreApplication.translate('Processing', string)

class ContourGeneratorAlgorithmError( RuntimeError ):
    pass

class ContourGeneratorAlgorithm(QgsProcessingAlgorithm):
    """
    Algorithm to calculate contour lines or filled contours from
    attribute values of a point data layer.  
    """

    # Constants used to refer to parameters and outputs. They will be
    # used when calling the algorithm from another algorithm, or when
    # calling from the QGIS console.

    PrmOutputLayer = 'OutputLayer'
    PrmInputLayer = 'InputLayer'
    PrmInputField = 'InputField'
    PrmContourMethod = 'ContourMethod'
    PrmNContour= 'NContour'
    PrmMinContourValue = 'MinContourValue'
    PrmMaxContourValue = 'MaxContourValue'
    PrmContourInterval = 'ContourInterval'
    PrmContourLevels = 'ContourLevels'
    PrmContourType = 'ContourType'
    PrmExtendContour = 'ExtendOption'
    PrmLabelDecimalPlaces = 'LabelDecimalPlaces'
    PrmLabelTrimZeros = 'LabelTrimZeros'
    PrmLabelUnits = 'LabelUnits'
    PrmDuplicatePointTolerance = 'DuplicatePointTolerance'

    TypeValues=ContourType.types()
    TypeOptions=[ContourType.description(t) for t in TypeValues]

    ExtendValues=ContourExtendOption.options()
    ExtendOptions=[ContourExtendOption.description(t) for t in ExtendValues]

    MethodValues=[m.id for m in ContourMethod.methods]
    MethodOptions=[m.name for m in ContourMethod.methods]

    EnumMapping={
        PrmContourMethod: 
           (MethodValues,MethodOptions),
        PrmContourType: 
           (TypeValues,TypeOptions),
        PrmExtendContour: 
           (ExtendValues,ExtendOptions)
    }

    def _enumParameter(self,name,description,optional=False):
        values,options=self.EnumMapping[name]
        return QgsProcessingParameterEnum( name, description, options, optional=optional)

    def _getEnumValue(self, parameters, name, context):
        # Wishful thinking - currently enum parameter can only accept integer value :-(
        # Hopefully will be able to get value as code in the future
        values,options=self.EnumMapping[name]
        if name not in parameters:
            return values[0]
        id=self.parameterAsString(parameters,name,context)
        if id not in values:
            try:
                id=values[int(id)]
            except:
                raise ContourGeneratorAlgorithmError(
                        tr('Invalid value {0} for {1}').format(id,name))
        return id

    def initAlgorithm(self, config):
        """
        Set up parameters for the ContourGenerator algorithm
        """
        #Would be cleaner to create a widget, at least for the contour levels.

        # Add the input point vector features source. 
        # geometry.

        self.addParameter(
            QgsProcessingParameterFeatureSource(
                self.PrmInputLayer,
                tr('Input point layer'),
                [QgsProcessing.TypeVectorPoint]
            )
        )

        # Define the field/expression to contour

        self.addParameter(
            QgsProcessingParameterExpression(
                self.PrmInputField,
                tr('Value to contour'),
                parentLayerParameterName=self.PrmInputLayer
            )
        )

        # Duplicate point radius - discards points if closer than
        # this to each other (approximately).  0 means don't discard

        self.addParameter(
            QgsProcessingParameterNumber(
                self.PrmDuplicatePointTolerance,
                tr('Duplicate point tolerance'),
                QgsProcessingParameterNumber.Double,
                minValue=0.0,
                defaultValue=0.0,
                optional=True
                )
        )

        # Define the contour type

        self.addParameter(
            self._enumParameter(
                self.PrmContourType,
                tr("Contour type")
            ))

        self.addParameter(
            self._enumParameter(
                self.PrmExtendContour,
                tr("Extend filled contour options"),
                optional=True
            ))

        # Define the contour level calculation method

        self.addParameter(
            self._enumParameter(
                self.PrmContourMethod,
                tr('Method used to calculate the contour levels')
            ))

        self.addParameter(
            QgsProcessingParameterNumber(
                self.PrmNContour,
                tr('Number (or max number) of contours'),
                defaultValue=20,
                minValue=1,
                optional=True
            ))

        self.addParameter(
            QgsProcessingParameterNumber(
                self.PrmMinContourValue,
                tr('Minimum contour level (omit to use data minimum)'),
                type=QgsProcessingParameterNumber.Double,
                optional=True
            ))

        self.addParameter(
            QgsProcessingParameterNumber(
                self.PrmMaxContourValue,
                tr('Maximum contour level (omit to use data maximum)'),
                type=QgsProcessingParameterNumber.Double,
                optional=True
            ))

        self.addParameter(
            QgsProcessingParameterNumber(
                self.PrmContourInterval,
                tr('Contour interval'),
                QgsProcessingParameterNumber.Double,
                minValue=0.0,
                defaultValue=1.0,
                optional=True
                ))

        self.addParameter(
            QgsProcessingParameterString(
                self.PrmContourLevels,
                tr('User selected contour levels'),
                multiLine=True,
                optional=True
                ))

        # Define label formatting - number of significant digits and 
        # whether trailiing zeros are trimmed.

        self.addParameter(
            QgsProcessingParameterNumber(
                self.PrmLabelDecimalPlaces,
                tr('Label decimal places (-1 for auto)'),
                QgsProcessingParameterNumber.Integer,
                defaultValue=-1,
                minValue=-1,
                maxValue=10,
                optional=True
                )
        )

        self.addParameter(
            QgsProcessingParameterBoolean(
                self.PrmLabelTrimZeros,
                tr("Trim trailing zeros from labels"),
                False,
                optional=True
            ))

        self.addParameter(
            QgsProcessingParameterString(
                self.PrmLabelUnits,
                tr("Units to append to label values"),
                "",
                optional=True
            ))

        # Output layer for the contours

        self.addParameter(
            QgsProcessingParameterFeatureSink(
                self.PrmOutputLayer,
                tr('Output layer')
            )
        )

    def processAlgorithm(self, parameters, context, feedback):

        # Retrieve the contour parameters
        
        source = self.parameterAsSource(parameters, self.PrmInputLayer, context)
        field = self.parameterAsExpression( parameters, self.PrmInputField, context )
        DuplicatePointTolerance = self.parameterAsDouble( parameters, self.PrmDuplicatePointTolerance, context )
        
        method = self._getEnumValue( parameters, self.PrmContourMethod, context )

        ncontour = self.parameterAsInt( parameters, self.PrmNContour, context )
        zmin=None
        zmax=None
        if parameters[self.PrmMinContourValue] is not None:
            zmin = self.parameterAsDouble( parameters, self.PrmMinContourValue, context )
        if parameters[self.PrmMaxContourValue] is not None:
            zmax = self.parameterAsDouble( parameters, self.PrmMaxContourValue, context )
        interval = self.parameterAsDouble( parameters, self.PrmContourInterval, context )
        levels = self.parameterAsString( parameters, self.PrmContourLevels, context )

        contourtype = self._getEnumValue( parameters, self.PrmContourType, context )
        extend = self._getEnumValue( parameters, self.PrmExtendContour, context )
        labelndp = self.parameterAsInt( parameters, self.PrmLabelDecimalPlaces, context )
        labeltrim = self.parameterAsBool( parameters, self.PrmLabelTrimZeros, context )
        labelunits = self.parameterAsString( parameters, self.PrmLabelUnits, context )

        # Construct and configure the contour generator


        params={
            'min':zmin,
            'max':zmax,
            'ncontour':ncontour,
            'maxcontour':ncontour,
            'interval':interval,
            'levels':levels
            }   

        generator=ContourGenerator(source,field,feedback)
        generator.setDuplicatePointTolerance( DuplicatePointTolerance )
        generator.setContourMethod( method, params )
        generator.setContourType( contourtype )
        generator.setContourExtendOption( extend )
        generator.setLabelFormat( labelndp, labeltrim, labelunits )

        # Create the destination layer

        dest_id=None
        try:
            wkbtype=generator.wkbtype()
            fields=generator.fields()
            crs=generator.crs()

            (sink, dest_id) = self.parameterAsSink(parameters, self.PrmOutputLayer,
                    context, fields, wkbtype, crs )

            # Add features to the sink
            for feature in generator.contourFeatures():
                sink.addFeature(feature, QgsFeatureSink.FastInsert)

        except (ContourError,ContourMethodError) as ex:
            feedback.reportError(ex.message())

        return {self.PrmOutputLayer: dest_id}

    def icon(self):
        return QIcon(":/plugins/contour/contour.png")

    def name(self):
        return 'generatecontours'

    def helpUrl(self):
        file=os.path.realpath(__file__)
        file = os.path.join(os.path.dirname(file),'doc','ContourGeneratorAlgorithm.html')
        if not os.path.exists(file):
            return ''
        return QUrl.fromLocalFile(file).toString(QUrl.FullyEncoded)

    def displayName(self):
        return tr('Generate Contours')

    def shortHelpString(self):
        file=os.path.realpath(__file__)
        file = os.path.join(os.path.dirname(file),'ContourGeneratorAlgorithm.help')
        if not os.path.exists(file):
            return ''
        with open(file) as helpf:
            help=helpf.read()
        return help

    def group(self):
        return tr('Contouring')

    def groupId(self):
        return 'contouring'

    def createInstance(self):
        return ContourGeneratorAlgorithm()
