/*
 * Decompiled with CFR 0.152.
 */
package org.geowebcache.storage.blobstore.file;

import com.google.common.base.Preconditions;
import java.io.File;
import java.io.FileFilter;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.io.UnsupportedEncodingException;
import java.net.URLDecoder;
import java.nio.channels.FileChannel;
import java.nio.file.DirectoryStream;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import java.util.Collections;
import java.util.Iterator;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.atomic.AtomicLong;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import java.util.stream.StreamSupport;
import org.geotools.util.logging.Logging;
import org.geowebcache.GeoWebCacheException;
import org.geowebcache.config.ConfigurationException;
import org.geowebcache.filter.parameters.ParametersUtils;
import org.geowebcache.io.FileResource;
import org.geowebcache.io.Resource;
import org.geowebcache.mime.MimeException;
import org.geowebcache.mime.MimeType;
import org.geowebcache.storage.BlobStore;
import org.geowebcache.storage.BlobStoreListener;
import org.geowebcache.storage.BlobStoreListenerList;
import org.geowebcache.storage.CompositeBlobStore;
import org.geowebcache.storage.DefaultStorageFinder;
import org.geowebcache.storage.StorageException;
import org.geowebcache.storage.StorageObject;
import org.geowebcache.storage.TileObject;
import org.geowebcache.storage.TileRange;
import org.geowebcache.storage.blobstore.file.DefaultFilePathGenerator;
import org.geowebcache.storage.blobstore.file.FilePathGenerator;
import org.geowebcache.storage.blobstore.file.FilePathUtils;
import org.geowebcache.storage.blobstore.file.LayerMetadataStore;
import org.geowebcache.storage.blobstore.file.TempFileNameGenerator;
import org.geowebcache.storage.blobstore.file.TileFileVisitor;
import org.geowebcache.util.FileUtils;
import org.springframework.scheduling.concurrent.CustomizableThreadFactory;

public class FileBlobStore
implements BlobStore {
    private static Logger log = Logging.getLogger((String)FileBlobStore.class.getName());
    static final int DEFAULT_DISK_BLOCK_SIZE = 4096;
    public static final int BUFFER_SIZE = 32768;
    private final File stagingArea;
    private final String path;
    private int diskBlockSize = 4096;
    private final BlobStoreListenerList listeners = new BlobStoreListenerList();
    private FilePathGenerator pathGenerator;
    private File tmp;
    private ExecutorService deleteExecutorService;
    private LayerMetadataStore layerMetadata;
    private TempFileNameGenerator tmpGenerator = new TempFileNameGenerator();
    static final int paramIdLength = ParametersUtils.getId(Collections.singletonMap("A", "B")).length();

    public FileBlobStore(DefaultStorageFinder defStoreFinder) throws StorageException, ConfigurationException {
        this(defStoreFinder.getDefaultPath());
    }

    public FileBlobStore(String rootPath) throws StorageException {
        this(rootPath, new DefaultFilePathGenerator(rootPath));
    }

    public FileBlobStore(String rootPath, FilePathGenerator pathGenerator) throws StorageException {
        this.path = rootPath;
        this.pathGenerator = pathGenerator;
        File fh = new File(this.path);
        fh.mkdirs();
        if (!(fh.exists() && fh.isDirectory() && fh.canWrite())) {
            throw new StorageException(this.path + " is not writable directory.");
        }
        boolean exists = new File(fh, "metadata.properties").exists();
        boolean empty = true;
        try (DirectoryStream<Path> ds = Files.newDirectoryStream(fh.toPath());){
            Iterator<Path> iterator = ds.iterator();
            if (iterator.hasNext()) {
                Path p = iterator.next();
                empty = false;
            }
        }
        catch (StorageException e) {
            throw e;
        }
        catch (IOException e) {
            throw new StorageException("Error while checking that " + rootPath + " is empty", e);
        }
        CompositeBlobStore.checkSuitability(rootPath, exists, empty);
        this.tmp = new File(this.path, "tmp");
        try {
            Files.createDirectories(this.tmp.toPath(), new FileAttribute[0]);
        }
        catch (IOException e) {
            throw new StorageException(this.tmp.getPath() + " is not writable directory.", e);
        }
        File metadataFile = new File(this.path, "metadata.properties");
        try {
            metadataFile.createNewFile();
        }
        catch (IOException e) {
            log.log(Level.SEVERE, "Error while writing blobstore metadata file " + metadataFile.getPath(), e);
        }
        this.stagingArea = new File(this.path, "_gwc_in_progress_deletes_");
        this.layerMetadata = new LayerMetadataStore(this.path, this.tmp);
        this.createDeleteExecutorService();
        this.issuePendingDeletes();
    }

    private void issuePendingDeletes() {
        File[] pendings;
        if (!this.stagingArea.exists()) {
            return;
        }
        if (!this.stagingArea.isDirectory() || !this.stagingArea.canWrite()) {
            throw new IllegalStateException("Staging area is not writable or is not a directory: " + this.stagingArea.getAbsolutePath());
        }
        for (File directory : pendings = FileUtils.listFilesNullSafe(this.stagingArea)) {
            if (!directory.isDirectory()) continue;
            this.deletePending(directory);
        }
    }

    private void deletePending(File pendingDeleteDirectory) {
        this.deleteExecutorService.submit(new DefferredDirectoryDeleteTask(pendingDeleteDirectory));
    }

    private void createDeleteExecutorService() {
        CustomizableThreadFactory tf = new CustomizableThreadFactory("GWC FileStore delete directory thread-");
        tf.setDaemon(true);
        tf.setThreadPriority(1);
        this.deleteExecutorService = Executors.newFixedThreadPool(1);
    }

    @Override
    public void destroy() {
        if (this.deleteExecutorService != null) {
            this.deleteExecutorService.shutdown();
        }
    }

    @Override
    public boolean delete(String layerName) throws StorageException {
        File source = this.getLayerPath(layerName);
        String target = FilePathUtils.filteredLayerName(layerName);
        boolean ret = this.stageDelete(source, target);
        this.listeners.sendLayerDeleted(layerName);
        return ret;
    }

    private boolean stageDelete(File source, String targetName) throws StorageException {
        if (!source.exists() || !source.canWrite()) {
            log.info(source + " does not exist or is not writable");
            return false;
        }
        if (!this.stagingArea.exists() && !this.stagingArea.mkdirs()) {
            throw new StorageException("Can't create staging directory for deletes: " + this.stagingArea.getAbsolutePath());
        }
        File tmpFolder = new File(this.stagingArea, targetName);
        int tries = 0;
        while (tmpFolder.exists()) {
            String dirName = FilePathUtils.filteredLayerName(targetName + "." + ++tries);
            tmpFolder = new File(this.stagingArea, dirName);
        }
        boolean renamed = FileUtils.renameFile(source, tmpFolder);
        if (!renamed) {
            throw new IllegalStateException("Can't rename " + source.getAbsolutePath() + " to " + tmpFolder.getAbsolutePath() + " for deletion");
        }
        this.deletePending(tmpFolder);
        return true;
    }

    @Override
    public boolean deleteByGridsetId(String layerName, String gridSetId) throws StorageException {
        File[] gridSubsetCaches;
        File layerPath = this.getLayerPath(layerName);
        if (!layerPath.exists() || !layerPath.canWrite()) {
            log.info(layerPath + " does not exist or is not writable");
            return false;
        }
        String filteredGridSetId = FilePathUtils.filteredGridSetId(gridSetId);
        FileFilter filter = pathname -> {
            if (!pathname.isDirectory()) {
                return false;
            }
            String dirName = pathname.getName();
            return dirName.startsWith(filteredGridSetId);
        };
        for (File gridSubsetCache : gridSubsetCaches = FileUtils.listFilesNullSafe(layerPath, filter)) {
            String target = FilePathUtils.filteredLayerName(layerName) + "_" + gridSubsetCache.getName();
            this.stageDelete(gridSubsetCache, target);
        }
        this.listeners.sendGridSubsetDeleted(layerName, gridSetId);
        return true;
    }

    @Override
    public boolean rename(String oldLayerName, String newLayerName) throws StorageException {
        File oldLayerPath = this.getLayerPath(oldLayerName);
        File newLayerPath = this.getLayerPath(newLayerName);
        if (newLayerPath.exists()) {
            throw new StorageException("Can't rename layer directory " + oldLayerPath + " to " + newLayerPath + ". Target directory already exists");
        }
        if (!oldLayerPath.exists()) {
            this.listeners.sendLayerRenamed(oldLayerName, newLayerName);
            return true;
        }
        if (!oldLayerPath.canWrite()) {
            log.info(oldLayerPath + " is not writable");
            return false;
        }
        boolean renamed = FileUtils.renameFile(oldLayerPath, newLayerPath);
        if (!renamed) {
            throw new StorageException("Couldn't rename layer directory " + oldLayerPath + " to " + newLayerPath);
        }
        this.listeners.sendLayerRenamed(oldLayerName, newLayerName);
        return renamed;
    }

    private File getLayerPath(String layerName) {
        String prefix = this.path + File.separator + FilePathUtils.filteredLayerName(layerName);
        File layerPath = new File(prefix);
        return layerPath;
    }

    @Override
    public boolean delete(TileObject stObj) throws StorageException {
        boolean exists;
        File fh = this.getFileHandleTile(stObj, null);
        boolean ret = false;
        long length = fh.length();
        boolean bl = exists = length > 0L;
        if (exists) {
            if (!fh.delete()) {
                throw new StorageException("Unable to delete " + fh.getAbsolutePath());
            }
            stObj.setBlobSize((int)this.padSize(length));
            this.listeners.sendTileDeleted(stObj);
            ret = true;
        } else {
            log.finer("delete unexistant file " + fh.toString());
        }
        File parentDir = fh.getParentFile();
        parentDir.delete();
        return ret;
    }

    @Override
    public boolean delete(TileRange trObj) throws StorageException {
        String prefix = this.path + File.separator + FilePathUtils.filteredLayerName(trObj.getLayerName());
        File layerPath = new File(prefix);
        if (!layerPath.exists()) {
            return true;
        }
        if (!layerPath.isDirectory() || !layerPath.canWrite()) {
            throw new StorageException(prefix + " does is not a directory or is not writable.");
        }
        final String layerName = trObj.getLayerName();
        final String gridSetId = trObj.getGridSetId();
        final String blobFormat = trObj.getMimeType().getFormat();
        final String parametersId = trObj.getParametersId();
        final AtomicLong count = new AtomicLong();
        this.pathGenerator.visitRange(layerPath, trObj, new TileFileVisitor(){

            @Override
            public void visitFile(File tile, long x, long y, int z) {
                long length = tile.length();
                boolean deleted = tile.delete();
                if (deleted) {
                    FileBlobStore.this.listeners.sendTileDeleted(layerName, gridSetId, blobFormat, parametersId, x, y, z, FileBlobStore.this.padSize(length));
                    count.incrementAndGet();
                }
            }

            @Override
            public void postVisitDirectory(File dir) {
                dir.delete();
            }
        });
        log.info("Truncated " + count + " tiles");
        return true;
    }

    @Override
    public boolean get(TileObject stObj) throws StorageException {
        File fh = this.getFileHandleTile(stObj, null);
        if (!fh.exists()) {
            stObj.setStatus(StorageObject.Status.MISS);
            return false;
        }
        Resource resource = this.readFile(fh);
        stObj.setBlob(resource);
        stObj.setCreated(resource.getLastModified());
        stObj.setBlobSize((int)resource.getSize());
        return true;
    }

    @Override
    public void put(TileObject stObj) throws StorageException {
        Runnable upm = () -> this.persistParameterMap(stObj);
        File fh = this.getFileHandleTile(stObj, upm);
        long oldSize = fh.length();
        boolean existed = oldSize > 0L;
        this.writeFile(fh, stObj, existed);
        if (stObj.getCreated() > 0L) {
            try {
                fh.setLastModified(stObj.getCreated());
            }
            catch (Exception e) {
                log.log(Level.FINE, "Failed to set the last modified time to match the tile request time", e);
            }
        }
        stObj.setBlobSize((int)this.padSize(stObj.getBlobSize()));
        if (existed) {
            this.listeners.sendTileUpdated(stObj, this.padSize(oldSize));
        } else {
            this.listeners.sendTileStored(stObj);
        }
    }

    private File getFileHandleTile(TileObject stObj, Runnable onTileFolderCreation) throws StorageException {
        File tilePath;
        MimeType mimeType;
        try {
            mimeType = MimeType.createFromFormat(stObj.getBlobFormat());
        }
        catch (MimeException me) {
            log.log(Level.SEVERE, me.getMessage());
            throw new RuntimeException(me);
        }
        try {
            tilePath = this.pathGenerator.tilePath(stObj, mimeType);
        }
        catch (GeoWebCacheException e) {
            throw new StorageException("Failed to compute file path", e);
        }
        if (onTileFolderCreation != null) {
            log.fine("Creating parent tile folder and updating ParameterMap");
            File parent = tilePath.getParentFile();
            this.mkdirs(parent, stObj);
            onTileFolderCreation.run();
        }
        return tilePath;
    }

    private Resource readFile(File fh) throws StorageException {
        if (!fh.exists()) {
            return null;
        }
        return new FileResource(fh);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void writeFile(File target, TileObject stObj, boolean existed) throws StorageException {
        this.tmp.mkdirs();
        File temp = new File(this.tmp, this.tmpGenerator.newName());
        try {
            try (FileOutputStream fos = new FileOutputStream(temp);
                 FileChannel channel = fos.getChannel();){
                stObj.getBlob().transferTo(channel);
            }
            catch (IOException ioe) {
                throw new StorageException(ioe.getMessage() + " for " + target.getAbsolutePath());
            }
            if (FileUtils.renameFile(temp, target)) {
                temp = null;
            } else if (existed && target.delete() && FileUtils.renameFile(temp, target)) {
                temp = null;
            }
        }
        finally {
            if (temp != null) {
                log.warning("Tile " + target.getPath() + " was already written by another thread/process");
                temp.delete();
            }
        }
    }

    protected void persistParameterMap(TileObject stObj) {
        if (Objects.nonNull(stObj.getParametersId())) {
            this.putLayerMetadata(stObj.getLayerName(), "parameters." + stObj.getParametersId(), ParametersUtils.getKvp(stObj.getParameters()));
        }
    }

    @Override
    public void clear() throws StorageException {
        throw new StorageException("Not implemented yet!");
    }

    @Override
    public void addListener(BlobStoreListener listener) {
        this.listeners.addListener(listener);
    }

    @Override
    public boolean removeListener(BlobStoreListener listener) {
        return this.listeners.removeListener(listener);
    }

    private boolean mkdirs(File path, TileObject stObj) {
        if (path.exists()) {
            return false;
        }
        if (path.mkdir()) {
            return true;
        }
        String parentDir = path.getParent();
        if (parentDir == null) {
            return false;
        }
        this.mkdirs(new File(parentDir), stObj);
        return path.mkdir();
    }

    @Override
    public String getLayerMetadata(String layerName, String key) {
        try {
            return this.layerMetadata.getEntry(layerName, key);
        }
        catch (IOException e) {
            log.fine("Optimistic read of metadata key failed");
            return null;
        }
    }

    private static String urlDecUtf8(String value) {
        try {
            value = URLDecoder.decode(value, "UTF-8");
        }
        catch (UnsupportedEncodingException e) {
            throw new RuntimeException(e);
        }
        return value;
    }

    @Override
    public void putLayerMetadata(String layerName, String key, String value) {
        try {
            this.layerMetadata.putEntry(layerName, key, value);
        }
        catch (UnsupportedEncodingException e) {
            throw new RuntimeException(e);
        }
        catch (IOException e) {
            log.fine("Optimistic read of metadata during writing process failed");
        }
    }

    @Override
    public boolean layerExists(String layerName) {
        return this.getLayerPath(layerName).exists();
    }

    public void setBlockSize(int fileSystemBlockSize) {
        Preconditions.checkArgument((fileSystemBlockSize > 0 ? 1 : 0) != 0);
        this.diskBlockSize = fileSystemBlockSize;
    }

    private long padSize(long fileSize) {
        int blockSize = this.diskBlockSize;
        long actuallyUsedStorage = blockSize * (int)Math.ceil((double)fileSize / (double)blockSize);
        return actuallyUsedStorage;
    }

    @Override
    public boolean deleteByParametersId(String layerName, String parametersId) throws StorageException {
        File[] parameterCaches;
        File layerPath = this.getLayerPath(layerName);
        if (!layerPath.exists() || !layerPath.canWrite()) {
            log.info(layerPath + " does not exist or is not writable");
            return false;
        }
        for (File parameterCache : parameterCaches = FileUtils.listFilesNullSafe(layerPath, pathname -> {
            if (!pathname.isDirectory()) {
                return false;
            }
            String dirName = pathname.getName();
            return dirName.endsWith(parametersId);
        })) {
            String target = FilePathUtils.filteredLayerName(layerName) + "_" + parameterCache.getName();
            this.stageDelete(parameterCache, target);
        }
        this.listeners.sendParametersDeleted(layerName, parametersId);
        return true;
    }

    private Stream<Path> layerChildStream(String layerName, DirectoryStream.Filter<Path> filter) throws IOException {
        File layerPath = this.getLayerPath(layerName);
        if (!layerPath.exists()) {
            return Stream.of(new Path[0]);
        }
        DirectoryStream<Path> layerDirStream = Files.newDirectoryStream(layerPath.toPath(), filter);
        return (Stream)StreamSupport.stream(layerDirStream.spliterator(), false).onClose(() -> {
            try {
                layerDirStream.close();
            }
            catch (IOException e) {
                throw new UncheckedIOException(e);
            }
        });
    }

    public boolean isParameterIdCached(String layerName, String parametersId) throws IOException {
        try (Stream<Path> layerChildStream = this.layerChildStream(layerName, p -> Files.isDirectory(p, new LinkOption[0]) && p.endsWith(parametersId));){
            boolean bl = layerChildStream.findAny().isPresent();
            return bl;
        }
    }

    @Override
    public Map<String, Optional<Map<String, String>>> getParametersMapping(String layerName) {
        Map<String, String> p;
        try {
            p = this.layerMetadata.getLayerMetadata(layerName);
        }
        catch (IOException e) {
            log.fine("Optimistic read of metadata mappings failed");
            return null;
        }
        return this.getParameterIds(layerName).stream().collect(Collectors.toMap(id -> id, id -> {
            String kvp = (String)p.get("parameters." + id);
            if (Objects.isNull(kvp)) {
                return Optional.empty();
            }
            kvp = FileBlobStore.urlDecUtf8(kvp);
            return Optional.of(ParametersUtils.getMap(kvp));
        }));
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    @Override
    public Set<String> getParameterIds(String layerName) {
        try (Stream<Path> layerChildStream = this.layerChildStream(layerName, p -> Files.isDirectory(p, new LinkOption[0]));){
            Set<String> set = layerChildStream.map(p -> p.getFileName().toString()).map(s -> s.substring(s.lastIndexOf(95) + 1)).filter(s -> s.length() == paramIdLength).collect(Collectors.toSet());
            return set;
        }
        catch (IOException e) {
            throw new UncheckedIOException(e);
        }
    }

    private static class DefferredDirectoryDeleteTask
    implements Runnable {
        private final File directory;

        public DefferredDirectoryDeleteTask(File directory) {
            this.directory = directory;
        }

        @Override
        public void run() {
            try {
                this.deleteDirectory(this.directory);
            }
            catch (IOException e) {
                log.log(Level.WARNING, "Exception occurred while deleting '" + this.directory.getAbsolutePath() + "'", e);
            }
            catch (InterruptedException e) {
                log.info("FileStore delete background service interrupted while deleting '" + this.directory.getAbsolutePath() + "'. Process will be resumed at next start up");
                Thread.currentThread().interrupt();
            }
        }

        private void deleteDirectory(File directory) throws IOException, InterruptedException {
            File[] files;
            if (!directory.exists()) {
                return;
            }
            if (Thread.interrupted()) {
                throw new InterruptedException();
            }
            for (File value : files = FileUtils.listFilesNullSafe(directory)) {
                if (Thread.interrupted()) {
                    throw new InterruptedException();
                }
                File file = value;
                if (file.isDirectory()) {
                    this.deleteDirectory(file);
                    continue;
                }
                if (file.delete() || !file.exists()) continue;
                throw new IOException("Unable to delete " + file.getAbsolutePath());
            }
            if (!directory.delete() && directory.exists()) {
                String message = "Unable to delete directory " + directory + ".";
                throw new IOException(message);
            }
        }
    }
}

